/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

import {Grid} from '@PSTypes/grid';
import GridMap from '@components/grid/grid-map';

const {$} = window;

/**
 * Class BulkActionSelectCheckboxExtension
 */
export default class BulkActionCheckboxExtension {
  /**
   * Extend grid with bulk action checkboxes handling functionality
   *
   * @param {Grid} grid
   */
  extend(grid: Grid): void {
    this.handleBulkActionCheckboxStatus(grid);
    this.handleBulkActionCheckboxSelect(grid);
    this.handleBulkActionSelectAllCheckbox(grid);
  }

  /**
   * Disable/Enable "Select all" button in the grid
   *
   * @param {Grid} grid
   *
   * @private
   */
  private handleBulkActionCheckboxStatus(grid: Grid) {
    const gridBulkActionSelectAll = grid.getContainer().find(GridMap.bulks.actionSelectAll);
    gridBulkActionSelectAll.prop(
      'disabled',
      grid.getContainer().find(GridMap.bulks.bulkActionCheckbox).length === 0,
    );
  }

  /**
   * Handles "Select all" button in the grid
   *
   * @param {Grid} grid
   *
   * @private
   */
  private handleBulkActionSelectAllCheckbox(grid: Grid) {
    grid.getContainer().on('change', GridMap.bulks.actionSelectAll, (e) => {
      const $checkbox = $(e.currentTarget);

      const isChecked = $checkbox.is(':checked');

      if (isChecked) {
        this.enableBulkActionsBtn(grid);
      } else {
        this.disableBulkActionsBtn(grid);
      }

      grid
        .getContainer()
        .find(GridMap.bulks.bulkActionCheckbox)
        .prop('checked', isChecked);
    });
  }

  /**
   * Handles each bulk action checkbox select in the grid
   *
   * @param {Grid} grid
   *
   * @private
   */
  private handleBulkActionCheckboxSelect(grid: Grid) {
    grid.getContainer().on('change', GridMap.bulks.bulkActionCheckbox, () => {
      const checkedRowsCount = grid
        .getContainer()
        .find(GridMap.bulks.checkedCheckbox).length;

      if (checkedRowsCount > 0) {
        this.enableBulkActionsBtn(grid);
      } else {
        this.disableBulkActionsBtn(grid);
      }
    });
  }

  /**
   * Enable bulk actions button
   *
   * @param {Grid} grid
   *
   * @private
   */
  private enableBulkActionsBtn(grid: Grid): void {
    grid
      .getContainer()
      .find(GridMap.bulks.bulkActionBtn)
      .prop('disabled', false);
  }

  /**
   * Disable bulk actions button
   *
   * @param {Grid} grid
   *
   * @private
   */
  private disableBulkActionsBtn(grid: Grid): void {
    grid
      .getContainer()
      .find(GridMap.bulks.bulkActionBtn)
      .prop('disabled', true);
  }
}
